import sys
import json
import requests
import folium
import re

# 検索する都道府県
pref = "京都府"
print(f'{pref}を囲む長方形範囲を検索')

# 都道府県コード(JIS X 0401より)
PrefId = {
    '北海道': 1, '青森県': 2, '岩手県': 3, '宮城県': 4, '秋田県': 5,
    '山形県': 6, '福島県': 7, '茨城県': 8, '栃木県': 9, '群馬県': 10,
    '埼玉県': 11, '千葉県': 12, '東京都': 13, '神奈川県': 14, '新潟県': 15,
    '富山県': 16, '石川県': 17, '福井県': 18, '山梨県': 19, '長野県': 20,
    '岐阜県': 21, '静岡県': 22, '愛知県': 23, '三重県': 24, '滋賀県': 25,
    '京都府': 26, '大阪府': 27, '兵庫県': 28, '奈良県': 29, '和歌山県': 30,
    '鳥取県': 31, '島根県': 32, '岡山県': 33, '広島県': 34, '山口県': 35,
    '徳島県': 36, '香川県': 37, '愛媛県': 38, '高知県': 39, '福岡県': 40,
    '佐賀県': 41, '長崎県': 42, '熊本県': 43, '大分県': 44, '宮崎県': 45,
    '鹿児島県': 46, '沖縄県': 47
}

# 都道府県境界を囲む長方形の緯度経度座標
# 国土交通省　国土数値情報　行政区域第1.0版を参考に作成
# https://nlftp.mlit.go.jp/ksj/jpgis/datalist/KsjTmplt-N03.html
PrefRect_LatLon = [ [(0.0,0,0),(0.0,0.0)], # 0:dummy
    [(41.333333,139.25),(45.6,149.05)], #  1:北海道
    [(40.166667,139.375),(41.583333,141.75)], #  2:青森県
    [(38.666667,140.625),(40.5,142.125)], #  3:岩手県
    [(37.75,140.25),(39.083333,141.75)], #  4:宮城県
    [(38.833333,139.625),(40.583333,141.0)], #  5:秋田県
    [(37.666667,139.5),(39.25,140.75)], #  6:山形県
    [(36.75,139.125),(38.0,141.125)], #  7:福島県
    [(35.666667,139.625),(37.0,140.875)], #  8:茨城県
    [(36.166667,139.25),(37.166667,140.375)], #  9:栃木県
    [(35.916667,138.375),(37.083333,139.75)], # 10:群馬県
    [(35.75,138.625),(36.333333,140.0)], # 11:埼玉県
    [(34.833333,139.625),(36.166667,141.0)], # 12:千葉県
    [(20.416667,136.0),(35.916667,154.0)], # 13:東京都
    [(35.083333,138.875),(35.75,139.875)], # 14:神奈川県
    [(36.666667,137.625),(38.583333,140.0)], # 15:新潟県
    [(36.25,136.75),(37.0,137.875)], # 16:富山県
    [(36.0,136.125),(37.916667,137.375)], # 17:石川県
    [(35.333333,135.375),(36.333333,136.875)], # 18:福井県
    [(35.166667,138.125),(36.0,139.25)], # 19:山梨県
    [(35.166667,137.25),(37.083333,138.75)], # 20:長野県
    [(35.083333,136.25),(36.5,137.75)], # 21:岐阜県
    [(34.5,137.375),(35.666667,139.25)], # 22:静岡県
    [(34.5,136.625),(35.5,137.875)], # 23:愛知県
    [(33.666667,135.75),(35.333333,137.0)], # 24:三重県
    [(34.75,135.75),(35.75,136.5)], # 25:滋賀県
    [(34.666667,134.75),(35.833333,136.125)], # 26:京都府
    [(34.25,135.0),(35.083333,135.75)], # 27:大阪府
    [(34.083333,134.25),(35.75,135.5)], # 28:兵庫県
    [(33.833333,135.5),(34.833333,136.25)], # 29:奈良県
    [(33.416667,134.875),(34.416667,136.125)], # 30:和歌山県
    [(35.0,133.125),(35.666667,134.625)], # 31:鳥取県
    [(34.25,131.625),(36.416667,133.5)], # 32:島根県
    [(34.25,133.25),(35.416667,134.5)], # 33:岡山県
    [(34.0,132.0),(35.166667,133.5)], # 34:広島県
    [(33.666667,130.75),(34.833333,132.5)], # 35:山口県
    [(33.5,133.625),(34.333333,134.875)], # 36:徳島県
    [(34.0,133.375),(34.583333,134.5)], # 37:香川県
    [(32.833333,132.0),(34.333333,133.75)], # 38:愛媛県
    [(32.666667,132.375),(33.916667,134.375)], # 39:高知県
    [(32.916667,130.0),(34.25,131.25)], # 40:福岡県
    [(32.916667,129.625),(33.666667,130.625)], # 41:佐賀県
    [(31.916667,128.0),(34.75,130.5)], # 42:長崎県
    [(32.083333,129.875),(33.25,131.375)], # 43:熊本県
    [(32.666667,130.75),(33.75,132.25)], # 44:大分県
    [(31.333333,130.625),(32.916667,132.0)], # 45:宮崎県
    [(27.0,128.375),(32.333333,131.25)], # 46:鹿児島県
    [(24.0,122.875),(27.916667,131.375)] # 47:沖縄県
]

# 検索する緯度経度範囲
range_latlon = PrefRect_LatLon[PrefId[pref]]
query_range_latlon = f'({range_latlon[0][0]},{range_latlon[0][1]}),({range_latlon[1][0]},{range_latlon[1][1]})'
print(f'範囲: {query_range_latlon}')

# 検索条件
query = {
    'database': [ "nijl_siryousyozaijyouhou_kensaku" ],
    'query': {
        'conditions': [
            {
                'query': {
                    'field': "spatial",
                    'term': query_range_latlon
                }
            }
        ]
    }
}

# API「メタデータ検索(HIT数)」を呼び出す
cmd = "https://api.bridge.nihu.jp/v1/integratedsearch/metadata/search-hits"
response = requests.post(cmd, json=query)
result = response.json()

# 検索結果を画面に出力する（pretty printing）
#print(json.dumps(result, indent=2, ensure_ascii=False))

# ヒット件数
n_hits = result['info']['total']
print(f'{n_hits}件ヒットしました')

# 「メタデータ検索」のレコード取得数の上限値
max_size = 1000

# 文字列sから緯度経度座標を読み取りリストを返す関数
def scanLatLonStr(s):
    s1 = re.sub(r'[()]', "", s) # かっこをとる
    s2 = s1.split(',') # コンマを区切りにして分割（文字列のリスト）
    # 浮動小数点数のリストに変換
    return list(map(float, s2))

# 集合 -> コンマ区切り文字列
def SetToStr(s):
    t = ""
    for u in list(s):
        if len(t) > 0:
            t = t + ", "
        t = t + u
    return t

# レコードから取得する情報を格納する変数
# 座標 -> ヒット数
p_hits = {}
# 座標 -> 地名表記集合（現地名）
p_names = {}
# 座標 -> 地名表記集合（旧地名）
p_oldnames = {}

# ヒット数が１以上ならレコードを取得する
if n_hits > 0:

    # 取得するレコード位置の先頭
    start = 0

    # 検索条件の追加：取得するフィールド
    query['query']['fields'] = [ "spatial" ]

    # レコード群を max_size 件づつ取得する
    while start < n_hits:
        # 検索条件の追加：取得するレコード範囲
        remains = n_hits - start # 残りレコード数
        query['query']['paging'] = {
            'start': start,
            'size': max_size if remains > max_size else remains
        }
        #print(query)

        # API「メタデータ検索」を呼び出す
        cmd = "https://api.bridge.nihu.jp/v1/integratedsearch/metadata/search"
        response = requests.post(cmd, json=query)
        result = response.json()

        # 検索結果を画面に出力する（pretty printing）
        #print(json.dumps(result, indent=2, ensure_ascii=False))

        # 取得データを変数に読み込む
        for r in result['hits']:
            # 座標の取得
            p = scanLatLonStr(r['fields'][0]['value'][0]['place'][0])
            k = (p[0], p[1])
            # 件数のカウント
            if not k in p_hits:
                p_hits[k] = 0
            p_hits[k] = p_hits[k] + 1
            # 地名情報の取得
            if not k in p_names:
                p_names[k] = set()
                p_oldnames[k] = set()
            d = r['fields'][0]['value'][0]['description']
            # 地名は要素数2の配列[現地名，旧地名]
            p_names[k].add(d[0])
            p_oldnames[k].add(d[1])
            # 次のレコード範囲へ
            start = start + max_size

    # データの表示
    print(f'マークの数: {len(p_hits)}')
    for k, v in p_hits.items():
        print(f'{k}: {v:4}: {SetToStr(p_oldnames[k])}({SetToStr(p_names[k])})')

# 都道府県の地図を表示
print('***地図の表示***')

# 地図の表示中心を求める
# range_latlon = PrefRect_LatLon[PrefId[pref]]
center_lat = (range_latlon[0][0] + range_latlon[1][0]) / 2
center_lon = (range_latlon[0][1] + range_latlon[1][1]) / 2
center_latlon = (center_lat,center_lon)
print(center_latlon, range_latlon)

# 地理院地図を用いて地図を作成する
prefmap = folium.Map(location=center_latlon,
                     zoom_start=6,
                     tiles="https://cyberjapandata.gsi.go.jp/xyz/std/{z}/{x}/{y}.png",
                     attr="地理院地図",
                     crs="EPSG3857",
                     width="100%", height="100%")

# 都道府県境界を囲む長方形を描画する
folium.Rectangle(bounds=range_latlon).add_to(prefmap)

# マーカーを描画する　ポップアップに情報を登録する
for k, v in p_hits.items():
    f = folium.IFrame(f'{SetToStr(p_oldnames[k])}<br/>{SetToStr(p_names[k])}<br/>{v}件')
    p = folium.Popup(f, min_width=200, max_width=200)
    folium.Marker(location=k,popup=p).add_to(prefmap)

# 地図表示の大きさを最適化する
prefmap.fit_bounds(bounds=range_latlon)

# 地図を表示する
prefmap
#prefmap.save("spatial_out.html")
